%% Step 1: Calculate atmospheric stability of Single-point sources - Using US EPA 2013 point source tables
clear all;
close all;

% Define the parent directory containing the folders
parentPath = '\\csunas01.colostate.edu\research\METEC\User\mbua\ADED\OTM33_Gaussian\Event Tables';

% Specify the folders to process
main_folders = {'MGGA'};
minute_folders = {'5 minutes', '15 minutes', '30 minutes'};
degree_folders = {'5 degrees', '10 degrees', '20 degrees'};

% Loop through each main folder
for f = 1:numel(main_folders)
    folderName = main_folders{f};
    
    % Loop through each minute folder
    for m = 1:numel(minute_folders)
        minuteFolder = minute_folders{m};
        
        % Loop through each degree folder
        for d = 1:numel(degree_folders)
            degreeFolder = degree_folders{d};
            
            % Define the file path for the current folder combination
            path = fullfile(parentPath, folderName, 'Emission Sources', 'Single_Point',minuteFolder, degreeFolder,'Downwind');
            
            % Check if the directory exists
            if ~exist(path, 'dir')
                warning('Directory not found: %s', path);
                continue;
            end
            
            % Get a list of all CSV files in the directory
            files = dir(fullfile(path, '*.csv'));
            
            % Skip if no files found
            if isempty(files)
                warning('No CSV files found in: %s', path);
                continue;
            end
            
            fprintf('Processing folder: %s\n', path);
            
            % Loop over each file
            for k = 1:numel(files)
                filename = fullfile(path, files(k).name);
                
                % Read data from the current file
                data = readtable(filename);
                
                % Calculate WD standard deviation
                data.std_WD = repmat(std(data.WD), height(data), 1);
                
                % Calculate turbulence intensity
                turb_int = std(data.WS)/mean(data.WS);
                data.turb_int = repmat(turb_int, height(data), 1);

                % A. Use WD to estimate stability
                WD_values = data.std_WD; 
                stability_WD_values = zeros(size(WD_values));
                
                for j = 1:numel(WD_values)
                    if (WD_values(j) > 27.50)
                        stability_WD_values(j) = 1;
                    elseif (WD_values(j) > 23.50) && (WD_values(j) <= 27.50)
                        stability_WD_values(j) = 2;
                    elseif (WD_values(j) > 19.50) && (WD_values(j) <= 23.50)
                        stability_WD_values(j) = 3;
                    elseif (WD_values(j) > 15.50) && (WD_values(j) <= 19.50)
                        stability_WD_values(j) = 4;
                    elseif (WD_values(j) > 11.50) && (WD_values(j) <= 15.50)
                        stability_WD_values(j) = 5;
                    elseif (WD_values(j) > 7.50) && (WD_values(j) <= 11.50)
                        stability_WD_values(j) = 6;
                    elseif (WD_values(j) <= 7.50)
                        stability_WD_values(j) = 7;
                    end
                end
                data.stability_WD = stability_WD_values;

                % B. Use turb_int to estimate stability
                turb_int_values = data.turb_int;  
                stability_turb_int_values = zeros(size(turb_int_values));
                
                for j = 1:numel(turb_int_values)
                    if (turb_int_values(j) > 0.205)
                        stability_turb_int_values(j) = 1;
                    elseif (turb_int_values(j) > 0.180) && (turb_int_values(j) <= 0.205)
                        stability_turb_int_values(j) = 2;
                    elseif (turb_int_values(j) > 0.155) && (turb_int_values(j) <= 0.180)
                        stability_turb_int_values(j) = 3;
                    elseif (turb_int_values(j) > 0.130) && (turb_int_values(j) <= 0.155)
                        stability_turb_int_values(j) = 4;
                    elseif (turb_int_values(j) > 0.105) && (turb_int_values(j) <= 0.130)
                        stability_turb_int_values(j) = 5;
                    elseif (turb_int_values(j) > 0.08) && (turb_int_values(j) <= 0.105)
                        stability_turb_int_values(j) = 6;
                    elseif (turb_int_values(j) <= 0.08)
                        stability_turb_int_values(j) = 7;
                    end
                end
                data.stability_turb_int = stability_turb_int_values;

                % C. Use pgi to estimate stability
                pgi_values = (data.stability_WD + data.stability_turb_int) / 2;  
                stability_pgi_values = zeros(size(pgi_values));
                
                for j = 1:numel(pgi_values)
                    if (pgi_values(j) < 1.5)
                        stability_pgi_values(j) = 1;
                    elseif (pgi_values(j) >= 1.5) && (pgi_values(j) < 2.5)
                        stability_pgi_values(j) = 2;
                    elseif (pgi_values(j) >= 2.5) && (pgi_values(j) < 3.5)
                        stability_pgi_values(j) = 3;
                    elseif (pgi_values(j) >= 3.5) && (pgi_values(j) < 4.5)
                        stability_pgi_values(j) = 4;
                    elseif (pgi_values(j) >= 4.5) && (pgi_values(j) < 5.5)
                        stability_pgi_values(j) = 5;
                    elseif (pgi_values(j) >= 5.5) && (pgi_values(j) < 6.5)
                        stability_pgi_values(j) = 6;
                    elseif (pgi_values(j) >= 6.5)
                        stability_pgi_values(j) = 7;
                    end
                end
                data.stability_pgi = stability_pgi_values;

                % Save the updated table back to the file
                writetable(data, filename);
                fprintf('Processed file: %s\n', files(k).name);
            end
        end
    end
end

disp('Atmospheric stability calculation complete for all specified folders.');

%% Step 2: Calculate Dispersion Coefficients from atmospheric stability - US EPA method-sigmaY and sigma are from the document
clear all;
close all;

% Define the parent directory containing the folders
parentPath = '\\csunas01.colostate.edu\research\METEC\User\mbua\ADED\OTM33_Gaussian\Event Tables';

% Specify the folders to process
main_folders = {'MGGA'};
minute_folders = {'5 minutes', '15 minutes', '30 minutes'};
degree_folders = {'5 degrees', '10 degrees', '20 degrees'};

% Load the Sigma_y and Sigma_z tables
sigmaYTable = readtable('\\csunas01.colostate.edu\research\METEC\User\mbua\ADED\OTM33_Gaussian\Sigma_y.xlsx');
sigmaZTable = readtable('\\csunas01.colostate.edu\research\METEC\User\mbua\ADED\OTM33_Gaussian\Sigma_z.xlsx');

% Loop through each main folder
for f = 1:numel(main_folders)
    folderName = main_folders{f};
    
    % Loop through each minute folder
    for m = 1:numel(minute_folders)
        minuteFolder = minute_folders{m};
        
        % Loop through each degree folder
        for d = 1:numel(degree_folders)
            degreeFolder = degree_folders{d};
            
            % Define the file path for the current folder combination
            path = fullfile(parentPath, folderName, 'Emission Sources', 'Multi_Point','Emission Point', minuteFolder, degreeFolder,'Downwind','Single_Emission');
            
            % Check if the directory exists
            if ~exist(path, 'dir')
                warning('Directory not found: %s', path);
                continue;
            end
            
            % Get a list of all CSV files in the directory
            files = dir(fullfile(path, '*.csv'));
            
            % Skip if no files found
            if isempty(files)
                warning('No CSV files found in: %s', path);
                continue;
            end
            
            fprintf('Processing folder: %s\n', path);
            
            % Loop over each file
            for k = 1:numel(files)
                filename = fullfile(path, files(k).name);
                
                % Read data from the current file
                data = readtable(filename);
                
                % Round off Distance_meters
                data.Distance_meters = round(data.Distance_meters);
                
                % Add new columns for sigmay and sigmaz
                data.sigmay = nan(height(data), 1);
                data.sigmaz = nan(height(data), 1);
                
                % Loop through each row in the data table
                for row = 1:height(data)
                    distance = data.Distance_meters(row);
                    stabilityClass = data.stability_pgi(row);  % Assumes 'stability_pgi' is the column name
                    
                    % Find the corresponding row in the sigmaYTable for the given distance
                    matchIdxY = find(sigmaYTable.Distance_meters == distance);
                    
                    if ~isempty(matchIdxY) && stabilityClass >= 1 && stabilityClass <= 7
                        % Retrieve the corresponding sigmay value
                        columnIdxY = stabilityClass + 1;  % Offset by 1 due to the first column being 'Distance_meters'
                        data.sigmay(row) = sigmaYTable{matchIdxY, columnIdxY};
                    end
                    
                    % Find the corresponding row in the sigmaZTable for the given distance
                    matchIdxZ = find(sigmaZTable.Distance_meters == distance);
                    
                    if ~isempty(matchIdxZ) && stabilityClass >= 1 && stabilityClass <= 7
                        % Retrieve the corresponding sigmaz value
                        columnIdxZ = stabilityClass + 1;  % Offset by 1 due to the first column being 'Distance_meters'
                        data.sigmaz(row) = sigmaZTable{matchIdxZ, columnIdxZ};
                    end
                end
                
                % Save the updated table back to the file
                writetable(data, filename);
                fprintf('Processed file: %s\n', files(k).name);
            end
        end
    end
end

disp('Dispersion coefficients calculation complete for all specified folders.');

%% Step 3:  Quantification
clear all;
close all;
clc;


non_release_data = readtable("\\csunas01.colostate.edu\research\METEC\User\mbua\ADED\CH4 MGGA\Background\non-release data.csv");
measurement_height = readtable("\\csunas01.colostate.edu\research\METEC\User\mbua\ADED\Releases\Measured_emission_points.csv");

% Convert time formats
non_release_data.Time = datetime(non_release_data.Time, 'InputFormat', 'MM/dd/yyyy HH:mm:ss');


% Filter non-release data
non_release_data = non_release_data(non_release_data.mean_x_CH4__ppm > 0,:);

% Define directories
parentPath = '\\csunas01.colostate.edu\research\METEC\User\mbua\ADED\OTM33_Gaussian\Event Tables\MGGA\Emission Sources\Single_Point';
outputPath = fullfile(parentPath, 'Quantified');
if ~exist(outputPath, 'dir')
    mkdir(outputPath);
end

minute_folders = {'5 minutes','15 minutes','30 minutes'}; % Keep your original time folders
degree_folders_to_process = {'5 degrees', '10 degrees', '20 degrees'}; % Added degree folders

for m = 1:numel(minute_folders)
    folderName = minute_folders{m};
    minutePath = fullfile(parentPath, folderName);
    
    % Process each specified degree folder
    for deg_idx = 1:numel(degree_folders_to_process)
        degreeFolder = degree_folders_to_process{deg_idx};
        degreePath = fullfile(minutePath, degreeFolder,'Downwind');
        
        % Skip if folder doesn't exist
        if ~exist(degreePath, 'dir')
            warning('Degree folder not found: %s', degreePath);
            continue;
        end
        
        outputDegreePath = fullfile(outputPath, folderName, degreeFolder);
        if ~exist(outputDegreePath, 'dir')
            mkdir(outputDegreePath);
        end
        
        files = dir(fullfile(degreePath, '*.csv'));
        
        for k = 1:numel(files)
            filename = fullfile(degreePath, files(k).name);
            data = readtable(filename);
            
            % Convert time format
            try
                data.Time = datetime(data.Time, 'InputFormat', 'MM/dd/yyyy HH:mm:ss');
            catch
                data.Time = datetime(data.Time);
            end
            
            if ismember('WS', data.Properties.VariableNames) && ...
               ismember('Uast', data.Properties.VariableNames) && ...
               ismember('L', data.Properties.VariableNames) && ...
               ismember('EmissionPoint', data.Properties.VariableNames)
                
                % 1. Match EmissionPoint to measurement height-added hs
                % later-may not be needed
                matched_idx = find(strcmp(measurement_height.EmissionPoint, data.EmissionPoint{1}));
                if ~isempty(matched_idx)
                    data.hs = repmat(measurement_height.hs(matched_idx), height(data), 1);
                else
                    warning('EmissionPoint not found for file: %s', files(k).name);
                    continue;
                end

  % the commented section is if sigmay and sigmaz are calculated from sonic anemometer data
                   % sigma_v = std(data.V);
                    %sigma_w = std(data.W);
           
                % Assign turbulence values
                %data.sigma_v = repmat(sigma_v, height(data), 1);
                %data.sigma_w = repmat(sigma_w, height(data), 1);
                
                % 3. Calculate dispersion coefficients
                
                %data.sigmay = data.sigma_v .* data.Distance_meters ./ data.mean_WS;
               % data.sigmaz = data.sigma_w .* data.Distance_meters ./ data.mean_WS;
    
                % 4. Boundary layer height calculation - needed if using
                % equation 8 (check SI)
                data.mean_WS = repmat(mean(data.WS), height(data), 1);
                L_values = data.L;
                h_values = zeros(size(L_values));
                for j = 1:numel(L_values)
                    if (L_values(j) <= 0) && (L_values(j) > -650)
                        h_values(j) = 2000;
                    elseif (L_values(j) <= -650) || (L_values(j) > 1000)
                        h_values(j) = 1000;
                    elseif (L_values(j) > 0) && (L_values(j) < 1000)
                        h_values(j) = 800;
                    end
                end
                data.h = h_values;
                
                % 5. Other parameters
                data.z = repmat(3, height(data), 1);
                data.meanAT = repmat(mean(data.AT), height(data), 1);
                data.y = data.Distance_meters .* sind(data.diff);
                
                % 6. Background concentration
                fileTime = data.Time(1);
                idx = find(non_release_data.Time <= fileTime, 1, 'last');
                
                if ~isempty(idx)
                    startTime = non_release_data.Time(idx) - minutes(5);
                    timeMask = non_release_data.Time >= startTime & non_release_data.Time <= non_release_data.Time(idx);
                    filteredData = non_release_data(timeMask, :);
                    
                    if ~isempty(filteredData)
                        percentile_5th = prctile(filteredData.mean_x_CH4__ppm, 5);
                        low_values = filteredData.mean_x_CH4__ppm(filteredData.mean_x_CH4__ppm <= percentile_5th);
                        Cb = mean(low_values);
                    else
                        Cb = NaN;
                    end
                    
                    mean_CH4_ppm = mean(data.x_CH4__ppm);
                    data.Cb = repmat(Cb, height(data), 1);
                    data.mean_CH4_ppm = repmat(mean_CH4_ppm, height(data), 1);
                    
                    if Cb > mean_CH4_ppm
                        data.Cb = repmat(min(data.x_CH4__ppm), height(data), 1);
                    end
                
                    % 7. Gaussian plume calculations - the Gaussian
                    % equations 7&8 gave same values
                    X_ppb = data.x_CH4__ppm .* 1000;
                    Enh_ppb = X_ppb - data.Cb * 1000;
                    data.Enh_g = Enh_ppb * 10^-9 * 12187 * 16.04 ./ data.meanAT;
                    mean_Enh_g = mean(data.Enh_g);
                    mean_y = mean(data.y);
                    
                    e0 = exp(-(mean_y.^2) ./ (2 * data.sigmay.^2));
                    e1 = exp(-(data.z - data.hs).^2 ./ (2 * data.sigmaz.^2));
                    e2 = exp(-(data.z + data.hs).^2 ./ (2 * data.sigmaz.^2));
                    e3 = exp(-(data.z - 2*data.h + data.hs).^2 ./ (2 * data.sigmaz.^2));
                    e4 = exp(-(data.z + 2*data.h - data.hs).^2 ./ (2 * data.sigmaz.^2));
                    e5 = exp(-(data.z - 2*data.h - data.hs).^2 ./ (2 * data.sigmaz.^2));
                    
                    data.exp = e0 .* (e1 + e2 + e3 + e4 + e5); 
                    data.pi2sigysigzU = 2 * pi .* data.mean_WS .* data.sigmay .* data.sigmaz;
                    data.Q_g_s = mean_Enh_g .* (data.pi2sigysigzU ./ data.exp);
                    data.Emission_kg_h = data.Q_g_s * 3600 * 0.001;
                    
                    % Save results
                    outputFilename = fullfile(outputDegreePath, files(k).name);
                    writetable(data, outputFilename);
                else
                    warning('No matching non-release data found for file: %s', files(k).name);
                end
            else
                warning('Required columns missing in file: %s', files(k).name);
            end
        end
    end
end

disp('Emissions quantification complete.');